(*-------------------------------------------------------------------------------------------*)
(* 1 : TYPES *)
(*-------------------------------------------------------------------------------------------*)

type mouvement1 =
	{mutable mv1: (int vect * int vect vect) list}
;;

type context =
	{mutable matrice: int vect vect}
;;

type repere = {mutable plan: int * int * int * int};;

type ops = OPS of (unit -> unit) * (unit -> unit) * (unit -> unit);;

type couleur = ORANGE | VERT | BLANC | ROUGE | BLEU | JAUNE | GRIS;;

type bouton = {titre: string;
		orx: int; ory: int;
		largeur: int; hauteur: int;
		mutable couleur: couleur;
		mutable action: unit -> unit;
		mutable bas: bool};;

type cube1 =
	{
		anime1: bool ref;
		mutable mouvement1: mouvement1;
		mutable mvi: mouvement1;
		mutable context1: context;
		mutable repere1: repere;
		mutable rotations_cube1: ops * ops;
		mutable rotations_faces1: ops * ops * ops * ops;
		mutable rotations_faces1i: ops * ops * ops * ops;
		mutable boutons1: bouton vect
	}
;;

type couple = COUPLE of (int vect * int vect) | NIL;;

(*-------------------------------------------------------------------------------------------*)
(* 2 : DIVERS *)
(*-------------------------------------------------------------------------------------------*)

let matrice_nulle = [|[|0; 0; 0|]; [|0; 0; 0|]; [|0; 0; 0|]|];;

let vect v = if vect_length v = 3 then (v.(0), v.(1), v.(2))
	else failwith "vect"
;;

let matscal a = let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			m.(i).(i) <- a
		done;
		m
;;

let id = matscal 1 and idm = matscal (- 1);;

(* produit du vecteur ligne entier v par la matrice entière m *)
let prefix /:/ v m =
	let w j = let t = ref 0 in for k = 0 to vect_length v - 1 do
				t := !t + m.(k).(j) * v.(k) done;
			!t in
		[|w 0; w 1; w 2|]
;;

(*produit du scalaire a par la matrice m*)
let prefix /../ a m =
	map_vect (fun x -> map_vect (fun t -> a * t) x) m;;

(* produit matriciel *)
let prefix /./ m m1 = map_vect (fun v -> v /:/ m1) m;;

(* somme matricielle *)
let prefix /+/ m1 m2 =
	let m = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m.(i).(j) <- m1.(i).(j) + m2.(i).(j)
			done;
		done;
		m
;;

(* matrice diagonale *)
let diag a b c = [|[|a; 0; 0|]; [|0; b; 0|]; [|0; 0; c|]|];;

(* transposée de la matrice m  qui en est aussi l'inverse : *)
(* quand m est orthogonale *)
let transpose m =
	let m1 = make_matrix 3 3 0 in
		for i = 0 to 2 do
			for j = 0 to 2 do
				m1.(j).(i) <- m.(i).(j)
			done;
		done;
		m1
;;

(* produit scalaire *)
let prefix /|/ v w = v.(0) * w.(0) + v.(1) * w.(1) + v.(2) * w.(2);;

(* matrices des rotations d'un quart de tour autour des axes : *)
(* (opèrent à droite sur les lignes) *)

(* sens des aiguilles d'une montre *)
let rot v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot"
;;

(* sens inverse des aiguilles d'une montre *)
let rot' v = match list_of_vect v with
		| [1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|]
		| [0; 1; 0] -> [|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|]
		| [0; 0; 1] -> [|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|]
		| [- 1; 0; 0] -> [|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|]
		| [0; - 1; 0] -> [|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|]
		| [0; 0; - 1] -> [|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|]
		| _ -> failwith "rot'"
;;

(* liste dans l'ordre des éléments de l satisfaisant 'critère' *)
let rec select critere l = match l with
		t :: r -> let l1 = select critere r in if critere t then t :: l1 else l1
		| _ -> []
;;

(* liste des entiers de 0 à n - 1 *)
let liste n =
	let v = make_vect n 0 in
		for i = 0 to n - 1 do
			v.(i) <- i
		done;
		list_of_vect v
;;

(* permutation aléatoire des éléments d'une liste l *)
let random_list l =
	let n = list_length l and l1 = ref []
	in
		for i = 0 to n - 1 do
			l1 := (vect_of_list (subtract l !l1)).(random__int (n - i)) :: !l1
		done;
		!l1
;;

(* signature de la permutation p des éléments de la liste l *)
let sign l p =
	let n = list_length l and v = vect_of_list l
	and m = ref 1 in
		for i = 0 to n - 1 do
			for j = i + 1 to n - 1 do
				let a = v.(i) and b = v.(j) in
					if p a > p b && b > a || p b > p a && a > b then m := - !m;
			done;
		done;
		!m
;;

(* exécution d'une liste de mouvements *)
let rec exe1 l =
	match l with
		t :: r -> t (); exe1 r;
		| [] -> ()
;;

let tete s =
	let l = string_length s in
		if l = 0 then ""
		else if l = 1 then s
		else if l = 2 then if s.[1] = `0` || s.[1] = `'` || s.[1] = `i` then s else sub_string s 0 1
		else match s.[1], s.[2] with
				| `i`, `'` -> sub_string s 0 3
				| `i`, _ -> sub_string s 0 2
				| `0`, `'` -> sub_string s 0 3
				| `0`, _ -> sub_string s 0 2
				| `'`, _ -> sub_string s 0 2
				| _ -> sub_string s 0 1
;;

let scinde s =
	let t = tete s and ls = string_length s in
		let lt = string_length t in
			let r = sub_string s lt (ls - lt) in
				(t, r)
;;

let rec op_names_from_string s =
	let (t, r) = scinde s in
		if r = "" then [t] else t :: op_names_from_string r
;;

let format_string op_names_string = "exec \"" ^ op_names_string ^ "\";;\n";;

let nbqdt n =
	if n = 0 then "0 quart de tour"
	else if n = 1 then "1 quart de tour"
	else printf__sprintf "%d quarts de tour" n
;;

(*-------------------------------------------------------------------------------------------*)
(* 3 : INDICES *)
(*-------------------------------------------------------------------------------------------*)

let indices =
	let l = ref [] in
		for i = 0 to 3 do
			for j = 0 to 3 do
				for k = 0 to 3 do
					l := [|2 * i - 3; 2 * j - 3; 2 * k - 3|] :: !l
				done
			done
		done;
		let f t = if t < 0 then - t else t in
			list_of_vect (matscal 3) @ list_of_vect (matscal (- 3)) @
			select (fun t -> f t.(0) > 1 || f t.(1) > 1 || f t.(2) > 1) !l
;;

let est_coin x = (x /|/ x) = 27;;
let est_angle x = (x /|/ x) = 19;;
let est_centre x = (x /|/ x) = 11;;
let est_axe x = (x /|/ x) = 9;;

(* liste des coins *)
let coins = select est_coin indices;;

(* liste des angles *)
let angles = select est_angle indices;;

(* liste des centres *)
let centres = select est_centre indices;;

(* liste des axes *)
let axes = select est_axe indices;;

(*-------------------------------------------------------------------------------------------*)
(* 4 : GROUPE DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

let groupe_du_cube =
	[
		[|[|1; 0; 0|]; [|0; 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 1; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; - 1; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; - 1|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|0; 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|1; 0; 0|]; [|0; 0; 1|]|];
		[|[|1; 0; 0|]; [|0; 0; 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; 1; 0|]; [|1; 0; 0|]|];
		[|[|0; 1; 0|]; [|- 1; 0; 0|]; [|0; 0; 1|]|];
		[|[|0; - 1; 0|]; [|- 1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|0; 1; 0|]; [|1; 0; 0|]; [|0; 0; - 1|]|];
		[|[|- 1; 0; 0|]; [|0; 0; - 1|]; [|0; - 1; 0|]|];
		[|[|0; 0; - 1|]; [|0; - 1; 0|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|0; - 1; 0|]; [|1; 0; 0|]|];
		[|[|- 1; 0; 0|]; [|0; 0; 1|]; [|0; 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; 1|]; [|1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|- 1; 0; 0|]; [|0; 1; 0|]|];
		[|[|0; 0; 1|]; [|- 1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; 1; 0|]; [|0; 0; - 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; - 1|]; [|1; 0; 0|]; [|0; - 1; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; - 1|]; [|1; 0; 0|]|];
		[|[|0; - 1; 0|]; [|0; 0; 1|]; [|- 1; 0; 0|]|];
		[|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
	]
;;

(* représentation des (inverses des éléments) du groupe par les éléments a0, d0 ,h0 *)
let decomposition r =
	let lops = [""; "a0a0"; "d0d0"; "h0h0"; "a0'"; "d0'"; "h0'"; "a0"; "d0"; "h0";
			"h0'a0a0"; "h0a0a0"; "d0d0a0"; "d0a0a0"; "d0'a0a0"; "d0d0a0'"; "h0a0";
			"d0a0'"; "d0'a0"; "h0a0'"; "d0a0"; "h0'a0'"; "h0'a0"; "d0'a0'"]
	in assoc r (map2 (fun x y -> (x, y)) groupe_du_cube lops)
;;


(*-------------------------------------------------------------------------------------------*)
(* 5 : GROUPE DES MOUVEMENTS *)
(*-------------------------------------------------------------------------------------------*)

(* groupe M des mouvements des minicubes *)

(* tri d'un mouvement selon l'ordre des indices *)
let trier mv1 = sort__sort (fun x y -> fst x < fst y) mv1;;

(* élément neutre de M *)
let e = map (fun x -> x, id) indices;;

(* conversion entre mouvement représenté par une fonction et mouvement *)
(* représenté par une liste : (int vect * int vect vect) list *)
let mv1_of_fun f =
	map (fun (x, y) -> (x, y /./ (f x))) e
;;
let fun_of_mv1 mv1 x =
	assoc x mv1
;;

(* mouvements globaux *)
let cst x = mv1_of_fun (fun t -> x);;

(* loi interne *)
let prefix /*/ mv1 mv1' =
	let f = fun_of_mv1 mv1 and f' = fun_of_mv1 mv1'
	in
		let s t = t /:/ (f t)
		in trier (mv1_of_fun (fun x -> (f x) /./ (f' (s x))))
;;

(* inverse d'un élément *)
let inverse mv1 = map (fun (x, y) -> (x /:/ y, transpose y)) mv1;;

(* mouvements de Rubik élémentaires *)

(* rotations dans le sens des aiguilles d'une montre d'un quart de tour de la *)
(* face - tranche interne dans le cas du cube 4x4 - normale au vecteur sortant 'v' *)
let rub v = mv1_of_fun
	(fun x -> if (x /|/ v) = 1 then rot v else id)
;;

(* mouvement inverse du précédent *)
let rub' v = inverse (rub v);;

(* mouvements de Rubik élémentaires *)

(* rotations dans le sens des aiguilles d'une montre d'un quart de tour de la *)
(* face normale au vecteur sortant v *)

let rub3 v = mv1_of_fun
	(fun x -> if (x /|/ v) = 3 then rot v else id)
;;

(* mouvement inverse du précédent *)
let rub3' v = inverse (rub3 v);;

(* enregistrement sur disque d'un mouvement: format portable *)
let enregistrer_mouv mv chemin =
	let rec aux mv =
		let traite x =
			printf__sprintf "%d%d%d" x.(0) x.(1) x.(2)
		in
			match mv with
				| [] -> ""
				| t :: r ->
							let (x, m) = t
							in
								traite x ^ traite m.(0) ^ traite m.(1) ^ traite m.(2);
								^ aux r
	in
		try
			let canalout = open_out chemin
			in
				output_string canalout (aux mv);
				close_out canalout
		with sys__Sys_error s -> failwith s
;;

(* lecture sur disque d'un mouvement : format portable *)
let couple_of_int_matrice s =
	let t = make_matrix 4 3 0 in
		for i = 0 to 3 do
			for j = 0 to 2 do
				t.(i).(j) <- s.(i * 3 + j)
			done
		done;
		(t.(0), [|t.(1); t.(2); t.(3)|])
;;
let int_vect s =
	let tete s =
		let l = string_length s in
			if l = 0 then ""
			else if s.[0] = `-` then sub_string s 0 2
			else sub_string s 0 1
	in
		let reste s =
			let l = string_length s
			and lt = string_length (tete s) in
				sub_string s lt (l - lt)
		in
			if s = "" then [||]
			else
				let rec aux ss =
					let t = tete ss and r = reste ss in
						if r <> "" then t :: aux r
						else [t]
				in vect_of_list (map int_of_string (aux s))
;;
let int_matrices_of_int_vect v =
	let lst = ref [] in
		for i = 0 to (vect_length v - 12) / 12 do
			lst := sub_vect v (12 * i) 12 :: !lst
		done;
		vect_of_list !lst
;;
let lire_mouv path =
	try
		let canalin = open_in path in
			let s = input_line canalin in
				close_in canalin;
				rev (list_of_vect (map_vect couple_of_int_matrice (int_matrices_of_int_vect (int_vect s))))
	with sys__Sys_error s1 -> print_string s1; e
;;

(*-------------------------------------------------------------------------------------------*)
(* 6 : TEST DE VALIDITÉ D'UN MOUVEMENT (APPARTENANCE AU SOUS-GROUPE DE RUBIK *)
(*-------------------------------------------------------------------------------------------*)

let marque x =
	if est_coin x then [|0; 0; x.(2)|]
	else [|0; 0; 0|]
;;

(* morphisme 's: M -> S' et section 'l: S -> M' *)
(* construction d'une section 'l' de la suite exacte '0 -> K -> M -> S -> 0' *)
(* En Caml on représente la sujection 's' par 'sur', la section 'l' par 'sec' et 'gij' par 'gg i j' *)

(* éléments g_{ij} alias gg i j de G servant à construire cette section *)
let gg i j =
	let critere i j g = if est_coin i && est_coin j then i /:/ g = j && marque i /:/ g = marque j else i /:/ g = j
	in
		hd (select (critere i j) groupe_du_cube)
		(* cette liste devrait toujours contenir exactement un élément *)
;;

(* décomposition 'm = ker m /*/ sec (sur m)' d'un mouvement 'm' *)
(* avec 'ker m' élément du noyau de 'sur' *)
(* 'p' pour 'permutation': 'p = sur m' est la permutation 'p' des indices associée au mouvement 'm' *)
let sec p = mv1_of_fun (fun i -> gg i (p i));;
let sur m = fun i -> i /:/ fun_of_mv1 m i;;
let ker m = m /*/ inverse (sec (sur m));;

(* stabilisateur du coin i *)
(* les stabilisateurs des angles et des centres sont triviaux *)
let st i =
	let m = [|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
	and j = [|3; 3; 3|]
	in
		if est_coin i then
			gg i j /./ m /./ gg j i
		else failwith "st"
;;

(* rotation totale des coins *)
let rtc m =
	let rtc_aux k = let f = fun_of_mv1 k in
			let indexc i = if f i = st i then 1
				else if f i = transpose (st i) then 2 else 0 in
				(list_it (prefix +) (map indexc coins) 0) mod 3
	in rtc_aux (ker m)
;;

(* test d'appartenance d'un mouvement au sous-groupe R *)
(* par nullité de la rotation totale des coins et égalité *)
(* des signatures des permutations des centres et des coins *)
(* voir aussi la fonction 'est_rubik' *)
let est_dans_R m = let p = sur m in
		sign centres p = sign coins p && rtc m = 0;;

(* mouvement général de type 'mv1' défini par les rotations de coins *)
(* et les permutations des centres (ou 'milieux'), des angles et des coins *)
let nouveau_mv1 pm pa pc ec =
	let k = mv1_of_fun
		(fun i ->
							if est_coin i then
								if ec i = 0 then id else if ec i = 1 then st i
								else transpose (st i)
							else id
		)
	and l = mv1_of_fun
		(
			fun i ->
							if est_angle i then
								gg i (pa i)
							else if est_coin i then gg i (pc i)
							else if est_centre i then gg i (pm i)
							else id
		)
	in k /*/ l
;;


(*- mouvement aléatoire -*)

(* permutation aléatoire d'une liste *)
let pl_r l =
	let l' = random_list l
	in fun i -> assoc i (map2 (fun x y -> x, y) l l')
;;

(* exposant aléatoire pour les coins *)
let ec_r = fun i -> if est_coin i then random__int 3 else failwith "ec_r";;

(* mouvement aléatoire général *)
let mv1_r () = nouveau_mv1 (pl_r centres) (pl_r angles) (pl_r coins) ec_r;;

(*- fin de mouvement général aléatoire de type mv1 -*)

(* mouvement de Rubik aléatoire *)
let mv1_rubik_r () =
	let rot_coin i n =
		nouveau_mv1 (fun x -> x) (fun x -> x) (fun x -> x) (fun j -> if j = i then n else 0)
	in
		let m = ref (mv1_r ()) in
			if rtc !m <> 0 then m := !m /*/ rot_coin [|3; 3; 3|] (3 - rtc !m);
			let p = sur !m in
				if sign coins p <> sign centres p then
					(
						let tr i j = sec (fun k -> if k = i then j else if k = j then i else k)
							(* mouvement de transposition de deux indices... *)
						in
							m := !m /*/ tr [|3; 3; 3|] [|3; - 3; 3|]
							(* ...appliqué à deux coins pour s'assurer que les permutations *)
							(* des coins et des centres ont même signature *)
					);
				!m
;;

(*-------------------------------------------------------------------------------------------*)
(* 7 : COULEURS *)
(*-------------------------------------------------------------------------------------------*)

(* couleur rvb de la  couleur c *)
let couleur_rvb_de_couleur c =
	match c with
		| ROUGE -> graphics__red
		| ORANGE -> graphics__rgb 255 165 0
		| BLEU -> graphics__rgb 0 150 225
		| VERT -> graphics__green
		| JAUNE -> graphics__yellow
		| BLANC -> graphics__white
		| GRIS -> graphics__rgb 100 100 100
;;

(* association entre couleurs et vecteurs normaux aux faces du cube *)
let couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> ORANGE
		| - 1, 0, 0 -> ROUGE
		| 0, 1, 0 -> VERT
		| 0, - 1, 0 -> BLEU
		| 0, 0, 1 -> BLANC
		| 0, 0, - 1 -> JAUNE
		| _ -> GRIS
;;

let couleur_rvb_de_face v =
	couleur_rvb_de_couleur (couleur_de_face v)
;;

let nom_couleur_de_face v =
	match vect v with
		| 1, 0, 0 -> "orange"
		| - 1, 0, 0 -> "rouge"
		| 0, 1, 0 -> "vert"
		| 0, - 1, 0 -> "bleu"
		| 0, 0, 1 -> "blanc"
		| 0, 0, - 1 -> "jaune"
		| _ -> "?"
;;

let nom_de_couleur couleur =
	match couleur with
		| ORANGE -> "ORANGE"
		| ROUGE -> "ROUGE"
		| VERT -> "VERT"
		| BLEU -> "BLEU"
		| BLANC -> "BLANC"
		| JAUNE -> "JAUNE"
		| _ -> "GRIS"
;;

(*-------------------------------------------------------------------------------------------*)
(* 8 : GRAPHISME *)
(*-------------------------------------------------------------------------------------------*)

let prj (ox, oy, ux, uy) v pt3 =
	let proj x y z =
		let c = sqrt 6. /. 2. in
			(c *. (y -. x) /. sqrt 2., c *. (-. (x +. y) +. 2. *. z) /. sqrt 6.)
	and (x, y, z) = vect (map_vect float_of_int pt3)
	in
		let (x1, y1, z1) =
			if v /|/ [|1; 1; 1|] = 1 then (x, y, z)
			else match vect v with
					| (_, 0, 0) -> (x -. 8., y, z)
					| (0, _, 0) -> (x, y -. 8., z)
					| _ -> (x, y, z -. 8.)
		in
			(
				int_of_float (float_of_int ox +. fst (proj x1 y1 z1) *. float_of_int ux),
				int_of_float (float_of_int oy +. snd (proj x1 y1 z1) *. float_of_int uy)
			)
;;

(* la fonction 'drawPoly' est utilisée pour tracer le pourtour des projections *)
(* des faces des minicubes *)
let drawPoly poly =
	let (x, y) = poly.(0) in graphics__moveto x y;
		for i = 1 to vect_length poly - 1 do
			let (x, y) = poly.(i) in graphics__lineto x y
		done;
		let (x, y) = poly.(0) in graphics__lineto x y;
;;

(* la fonction 'draw' est utilisée pour dessiner la projection 'x' d'une face *)
(* de minicube en superposant le tracé du pourtour à la couleur de remplissage *)
let draw x =
	let a, b = x in
		graphics__set_color b;
		graphics__fill_poly a;
		graphics__set_color graphics__black;
		drawPoly a
;;

(* 'face v c' renvoie, si le minicube à l'emplacement d'indice 'c' a une face F *)
(* dans la face du Rubik's cube normale au vecteur sortant 'v', sous forme de vecteur *)
(* une liste des 4 sommets de F correspondant à un parcours de son bord *)

let coeff = ref 1;;

let face v c =
	let e = v /|/ [|1; 1; 1|] in let w = [|e; e; e|] in
			let w1 = w /:/ rot v in
				let w2 = w1 /:/ rot v in
					let w3 = w2 /:/ rot v in
						let l = [w; w1; w2; w3] in
							let add m = for i = 0 to 2 do m.(i) <- m.(i) + !coeff * c.(i) done
							in
								do_list add l;
								vect_of_list l;
;;

(* numérotation des centres *)

let numero c =
	let est_centre i = (i /|/ i) = 11
	in
		if est_centre c then string_of_int (assoc c
				[
					[|- 1; - 1; 3|], 1;
					[|- 1; 1; 3|], 2;
					[|1; 1; 3|], 3;
					[|1; - 1; 3|], 0;
					[|- 1; - 1; - 3|], 0;
					[|- 1; 1; - 3|], 3;
					[|1; 1; - 3|], 2;
					[|1; - 1; - 3|], 1;
					[|3; - 1; - 1|], 0;
					[|3; - 1; 1|], 1;
					[|3; 1; 1|], 2;
					[|3; 1; - 1|], 3;
					[|- 1; - 3; - 1|], 0;
					[|- 1; - 3; 1|], 1;
					[|1; - 3; 1|], 2;
					[|1; - 3; - 1|], 3;
					[|- 3; - 1; - 1|], 1;
					[|- 3; - 1; 1|], 0;
					[|- 3; 1; 1|], 3;
					[|- 3; 1; - 1|], 2;
					[|- 1; 3; - 1|], 3;
					[|- 1; 3; 1|], 2;
					[|1; 3; 1|], 1;
					[|1; 3; - 1|], 0
				])
		else ""
;;

(* la fonction 'draw_n' est utilisée comme draw pour dessiner la projection 'x' d'une face *)
(* de minicube en superposant le tracé du pourtour à la couleur de remplissage *)
(* avec en plus inscription des numéros des centres dans leur position actuelle *)

let draw_n x =
	let a, b, c = x in
		graphics__set_color b;
		graphics__fill_poly a;
		graphics__set_color graphics__black;
		drawPoly a;
		let (* inscription des numéros actuels des centres *)
		((a1, b1), (a2, b2), (a3, b3), (a4, b4)) = (a.(0), a.(1), a.(2), a.(3))
		in
			let (c1, c2) = ((a1 + a3) / 2, (b1 + b3) / 2)
			in
				graphics__moveto c1 c2;
				graphics__draw_string c

;;

(* 'draw1' inscrit les numéros des emplacements usine des centres en gris *)

let draw1_n x =
	let a, _, c = x and abs x = if x > 0 then x else - x in
		graphics__set_color (graphics__rgb 100 100 100);
		let
		((a1, b1), (a2, b2), (a3, b3), (a4, b4)) = (a.(0), a.(1), a.(2), a.(3))
		in
			let (c1, c2) = ((a1 + a3 - abs (a2 - a3) + 4) / 2, (b1 + b3 - abs (b2 - b3) + 4) / 2)
			in
				graphics__moveto c1 c2;
				graphics__draw_string c

;;

(* 'faces' renvoie une liste de triplets : la première composante est un centre 'c', la deuxième composante *)
(* est un vecteur listant les 3 vecteurs unitaires sortants normaux aux faces visibles du minicube centré en 'c' *)
(* et la troisième est un vecteur dont la composante numéro i est un vecteur listant les 4 sommets de la face visible *)
(* normale au vecteur numéro i précédent : i = 0,1,2 pour un coin, i = 0,1 pour un angle, i = 0 pour un centre *)

let faces c =
	let d = vect_of_list (subtract (list_of_vect (diag (c.(0) / 3) (c.(1) / 3) (c.(2) / 3))) [[|0; 0; 0|]]) in
		c, d, map_vect (fun v -> face v c) d
;;

let affiche1 plan mat context centre largeur =
	let p = context.matrice and (ox, _, _, _) = plan in
		let c, d, f = faces centre in
			for i = 0 to vect_length d - 1 do
				let v = d.(i) /:/ mat in
					let g = map_vect (fun x -> x /:/ mat) f.(i) in
						draw_n ((map_vect (prj plan v)
								g),
							couleur_rvb_de_face (d.(i) /:/ transpose p), if ox > largeur / 2 then numero (c /:/ transpose p) else "");
						draw1_n ((map_vect (prj plan v)
								g),
							couleur_rvb_de_face (d.(i) /:/ transpose p), numero (c /:/ mat /:/ transpose p));
			done
;;

(* affichage du cube 4x4, avec numéros des centres, dans l'état mv *)
let affiche_mouvement plan context mv =
	let affiche plan mat context centre =
		let p = context.matrice in
			let c, d, f = faces centre in
				for i = 0 to vect_length d - 1 do
					let v = d.(i) /:/ mat in
						let g = map_vect (fun pt -> prj plan (v /:/ p) (pt /:/ p)) (map_vect (fun x -> x /:/ mat) f.(i))
						and coul = couleur_rvb_de_face d.(i)
						in
							draw_n (g, coul, numero c); (* numéro après mélange *)
							draw1_n (g, coul, numero (c /:/ mat)); (* numéro avant mélange *)
				done
	in
		let indices1 = select (fun t -> t /|/ t <> 9) indices in
			do_list (fun x -> affiche plan (fun_of_mv1 mv x) context x) indices1
;;

let dessine_cube cube = affiche_mouvement cube.repere1.plan cube.context1 cube.mouvement1.mv1;;

(*-------------------------------------------------------------------------------------------*)
(* 9 : OPÉRATIONS D'UN CUBE 4x4 *)
(*-------------------------------------------------------------------------------------------*)

let nbqt = ref 0
and lo = ref ""
and matr = ref id
and mctx = ref id;;

let nom_position_de_face v =
	match vect v with
		| (1, 0, 0) -> "a"
		| (0, 1, 0) -> "d"
		| (0, 0, 1) -> "h"
		| (- 1, 0, 0) -> "p"
		| (0, - 1, 0) -> "g"
		| (0, 0, - 1) -> "b"
		| _ -> failwith "nom_position_de_face"
;;

let associe mat s =
	let adh = map_vect nom_position_de_face (mat)
	and pgb = map_vect nom_position_de_face ((- 1) /../ mat)
	in
		match s with
			| "a" -> adh.(0)
			| "d" -> adh.(1)
			| "h" -> adh.(2)
			| "p" -> pgb.(0)
			| "g" -> pgb.(1)
			| "b" -> pgb.(2)
			| "a'" -> adh.(0) ^ "'"
			| "d'" -> adh.(1) ^ "'"
			| "h'" -> adh.(2) ^ "'"
			| "p'" -> pgb.(0) ^ "'"
			| "g'" -> pgb.(1) ^ "'"
			| "b'" -> pgb.(2) ^ "'"
			| "ai" -> adh.(0) ^ "i"
			| "di" -> adh.(1) ^ "i"
			| "hi" -> adh.(2) ^ "i"
			| "pi" -> pgb.(0) ^ "i"
			| "gi" -> pgb.(1) ^ "i"
			| "bi" -> pgb.(2) ^ "i"
			| "ai'" -> adh.(0) ^ "i'"
			| "di'" -> adh.(1) ^ "i'"
			| "hi'" -> adh.(2) ^ "i'"
			| "pi'" -> pgb.(0) ^ "i'"
			| "gi'" -> pgb.(1) ^ "i'"
			| "bi'" -> pgb.(2) ^ "i'"
			| _ -> failwith "associe"
;;

let operations cube =
	let (OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
	and (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) = cube.rotations_faces1 (*tranches externes*)
	and (OPS (ai, di, hi), OPS (ai', di', hi'), OPS (pi, gi, bi), OPS (pi', gi', bi')) = cube.rotations_faces1i (*tranches intermédiaires*)
	in
		
		let op_with_name s =
			let la = [("a", a); ("p", p); ("h", h); ("b", b); ("d", d); ("g", g);
					("a'", a'); ("p'", p'); ("h'", h'); ("b'", b'); ("d'", d'); ("g'", g');
					("ai", ai); ("pi", pi); ("hi", hi); ("bi", bi); ("di", di); ("gi", gi);
					("ai'", ai'); ("pi'", pi'); ("hi'", hi'); ("bi'", bi'); ("di'", di'); ("gi'", gi');
					("a0", a0); ("d0", d0); ("h0", h0); ("a0'", a0'); ("d0'", d0'); ("h0'", h0');]
			in
				assoc s la
		in
			let exec1 str = if str <> "" then
					let temp = cube.context1.matrice in
						let listop = op_names_from_string str in
							exe1 (map op_with_name listop);
							let temp1 = cube.context1.matrice in
								mctx := !mctx /./ transpose temp /./ temp1;
			and exec str = if str <> "" then
					let listop = op_names_from_string str in
						nbqt := !nbqt + list_length listop;
						if !mctx <> id then
							(
								lo := !lo ^ format_string (decomposition (transpose !mctx));
								mctx := id
							);
						exe1 (map op_with_name listop);
						lo := !lo ^ (format_string str);
						let m = (transpose cube.context1.matrice) /./ !matr in
							print_string (format_string (concat (map (associe m) listop)))
			in ((a0, d0, h0, a0', d0', h0', a, d, h, a', d', h', p, g, b, p',
						g', b', ai, di, hi, ai', di', hi', pi, gi, bi, pi', gi', bi'),
					(exec1, exec))
;;


(*-------------------------------------------------------------------------------------------*)
(* 10 : CUBE 3x3 SOUS JACENT : RÉSOLUTION PAR NIVEAUX: "supérieur, médian, inférieur" *)
(*-------------------------------------------------------------------------------------------*)

let psd_of_mv1_3 () = (* produit semi direct *)
	let indices () = let l = ref [] in
			for k = 1 downto - 1 do
				for j = 1 downto - 1 do
					for i = 1 downto - 1 do l := [|i; j; k|] :: !l
					done
				done
			done;
			subtract !l [[|0; 0; 0|]]
	and est_angle x = x /|/ x = 2
	and est_coin x = x /|/ x = 3
	in
		let marque x =
			if est_coin x then [|0; 0; x.(2)|] else if est_angle x then
				let a, b, c = x.(0), x.(1), x.(2)
				in match a, b, c with
						| 0, _, _ -> [|0; b; 0|]
						| _, 0, _ -> [|0; 0; c|]
						| _, _, 0 -> [|a; 0; 0|]
						| _ -> [|0; 0; 0|]
			else [|0; 0; 0|]
		in
			let gg i j =
				try
					let critere i j g = i /:/ g = j && marque i /:/ g = marque j
					in
						hd (select (critere i j) groupe_du_cube)
				with Failure x -> id
								(* cette liste devrait toujours contenir exactement un élément *)
			in
				let stc i =
					let x = [|1; 1; 1|]
					and m = [|[|0; 0; 1|]; [|1; 0; 0|]; [|0; 1; 0|]|]
					in
						gg i x /./ m /./ gg x i
				
				and sta i = let x = [|1; 0; 1|]
					and m = [|[|0; 0; 1|]; [|0; - 1; 0|]; [|1; 0; 0|]|]
					in
						gg i x /./ m /./ gg x i
				in
					let st i = if est_angle i then sta i
						else if est_coin i then stc i
						else failwith "st"
					and mv1_of_fun f =
						map (fun x -> (x, f x)) (indices ())
					and fun_of_mv1 mv1 x =
						assoc x mv1
					
					in
						let prefix /*/ mv1 mv1' =
							let f = fun_of_mv1 mv1 and f' = fun_of_mv1 mv1'
							in
								let s t = t /:/ (f t)
								in mv1_of_fun (fun x -> (f x) /./ (f' (s x)))
						and inverse mv1 = map (fun (x, y) -> (x /:/ y, transpose y)) mv1
						in
						(* relèvement d'un mouvement 'm' et décomposition 'm = ker m /*/ sec (sur m)' *)
						(* avec ker m élément du noyau de 'sur' *)
						(* 'p' pour 'permutation', par exemple 'p = sur m' *)
							let sec p = mv1_of_fun (fun i -> gg i (p i))
							and sur m = fun i -> i /:/ fun_of_mv1 m i
							in let ker m = m /*/ inverse (sec (sur m))
								in (sec, sur, ker, st)
;;

(* fournit le mouvement du cube 3x3 sous-jacent obtenu par appariement des angles... *)
let mv1_3_of_mv1_4 mv1_4 =
	let est_angle x = x /|/ x = 11
	in
		let encode a = [|a.(0) / 3; a.(1) / 3; a.(2) / 3|]
		and represente a = est_angle a && (a.(0) = 1 || a.(1) = 1 || a.(2) = 1) in
			map (fun (x, y) -> encode x, y) (select (fun (x, y) -> not (est_angle x) || represente x) mv1_4)
;;

(* ...en vue d'obtenir la rotation totale des angles correspondante *)
(* le paramètre mv1_4 est ici le mouvement de type mv1 du cube 4x4 *)
let rta mv1_4 =
	let (_, _, ker, st) = psd_of_mv1_3 ()
	and est_angle x = x /|/ x = 2
	and mv1_3 = mv1_3_of_mv1_4 mv1_4
	and fun_of_mv1 mv1 i = assoc i mv1
	in
		let rta_aux k = let f = fun_of_mv1 k in
				let ea i = if f i = st i then 1 else 0 in
					let angles = select est_angle (map fst k) in
						(list_it (prefix +) (map ea angles) 0) mod 2
		in rta_aux (ker mv1_3)
;;

(* signature de la permutation des angles d'un mouvement du sous-cube 3x3... *)
let sign_angles mv1 =
	let encode a = [|a.(0) / 3; a.(1) / 3; a.(2) / 3|]
	and decode a = let f x = if x = 0 then 1 else if x = 1 then 3 else - 3 in map_vect f a
	and represente a =
		let est_angle x = (x /|/ x) = 19 in
			est_angle a && (a.(0) = 1 || a.(1) = 1 || a.(2) = 1)
	in
		let perm x = encode (sur mv1 (decode x))
		and angles = map encode (select represente (map fst mv1))
		in
			sign angles perm
;;

(* ...en vue de corriger ce cube : la fonction fix_cube est destinée au troisième niveau du cube 3x3 *)
(* à n'utiliser que si le cube 3x3 issu du 4x4 n'a pas été déjà corrigé *)
let fix_cube cube =
	let pos1 x =
		let pp = cube.context1.matrice
		in
			transpose pp /./ (fun_of_mv1 (inverse cube.mouvement1.mv1)) (x /:/ transpose pp) /./ pp
	and (_, (exec1, exec)) = operations cube
	in
		let test_cube cube =
			let est_mal_oriente angle =
				let jumeau a =
					let f x = if x = 1 then - 1 else if x = - 1 then 1 else x in map_vect f a
				
				in
					let m = pos1 angle and m' = pos1 (jumeau angle) in
						m.(2) <> [|0; 0; 1|] && m'.(2) <> [|0; 0; 1|]
			in
				let v = map_vect est_mal_oriente
					[|[|3; 1; - 3|]; [|1; - 3; - 3|]; [|- 3; 1; - 3|]; [|1; 3; - 3|]|] in
					let coins = select (fun t -> t /|/ t = 27) indices
					in
						let n = list_it (fun a b -> if a = false then b + 1 else b) (list_of_vect v) 0 in
							(n mod 2), (sign_angles cube.mouvement1.mv1), (sign coins (sur (cube.mouvement1.mv1)))
		in
			let fix_flip () = exec1 "a0a0"; exec "didipphhgihhdi'hhdihhaadiaagi'ppdidi"; exec1 "a0a0"
			and fix_parity () = exec1 "a0a0"; exec "didihhdidihihihhdidihihi"; exec1 "a0a0"
			in
				let (n, sa, sc) = test_cube cube
				in
					if sa <> sc then fix_parity ();
					if n mod 2 <> 0 then fix_flip ();
;;

(*- nombres de quarts de tour des tranches externes modulo 4 -*)
let nqt1 mv1 x =
	let r = fun_of_mv1 mv1 x
	and encode x = [|x.(0) / 3; x.(1) / 3; x.(2) / 3|]
	in
		let v = encode x in
			if r = id then 0
			else if r = rot v then 1
			else if r = rot v /./ rot v then 2
			else if r = rot' v then 3
			else failwith "nqt"
;;

let nqt cube v =
	let p = cube.context1.matrice in
		let w = [|v.(0) * 3; v.(1) * 3; v.(2) * 3|] in
			nqt1 cube.mouvement1.mv1 (w /:/ transpose p)
;;
(*- fin de nombres de quarts de tour des tranches externes modulo 4 -*)


(* résolution par niveaux du cube 3x3 sous-jacent au 4x4 *)
(* ce cube 3x3 est obtenu en utilisant les fonctions 'regrouper_les_centres',*)
(* 'arranger_les_centres' et 'apparier_les_angles' définies plus loin *)

exception Orienter_les_coins;;
exception Placer_angle_frontal_haut;;
exception Descendre_coin;;
exception Remonter_coin;;
exception Remonter_angle;;
exception Orienter_les_angles;;
exception Placer_les_angles;;
exception Placer_les_coins;;

(*-------------------------------------------------*)

(* en repère adh, la matrice de passage dans le groupe du cube telle que dans le repère adh associé *)
(* le coin centré en x dans l'état mv ait les couleurs adh coul1, coul2, coul3 *)
let context_adh_aux (coul1, coul2, coul3) mv xx =
	let couleurs_adh context mouvement x =
		let eclate x = [|[|x.(0) / 3; 0; 0|]; [|0; x.(1) / 3; 0|]; [|0; 0; x.(2) / 3|]|] in
			let p = context.matrice in
				let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mouvement)) (x /:/ transpose p) in
					map_vect couleur_de_face m
	in
		{matrice = hd (select (fun p -> couleurs_adh {matrice = p} mv xx = [|coul1; coul2; coul3|]) groupe_du_cube)}
;;

let context_adh (coul1, coul2, coul3) mv = context_adh_aux (coul1, coul2, coul3) mv [|3; 3; 3|];;

(*-------------------------------------------------*)

(* les couleurs en repère adh des faces antérieure, droite et gauche du cube non mélangé *)
let couleurs_faces_adh cube =
	let _ = cube.context1.matrice in
		let feminin adjectif = match adjectif with
				| "blanc" -> "blanche"
				| "vert" -> "verte"
				| "bleu" -> "bleue"
				| _ -> adjectif
		in
			let noms = map_vect (fun x -> feminin (nom_couleur_de_face x)) (transpose cube.context1.matrice) in
				printf__sprintf "Résultat avec :\nface antérieure %s\nface droite %s \nface haute %s" noms.(0) noms.(1) noms.(2)
;;

(* les couleurs en repère adh des faces visibles du minicube centré en x *)
let couleurs_adh cube x =
	let eclate x = [|[|x.(0) / 3; 0; 0|]; [|0; x.(1) / 3; 0|]; [|0; 0; x.(2) / 3|]|] in
		let p = cube.context1.matrice and mv1 = cube.mouvement1.mv1 in
			let m = eclate x /./ transpose p /./ (fun_of_mv1 (inverse mv1)) (x /:/ transpose p) in
				map_vect couleur_de_face m
;;

(* les couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let couleurs_coin_adh cube = couleurs_adh cube [|3; 3; 3|];;

(* impression des noms des couleurs en repère adh des faces visibles du minicube centré en [|1; 1; 1|] *)
(* dans l'ordre antérieure, droite, haute *)
let noms_couleurs_coin_adh cube =
	let noms = map_vect nom_de_couleur (couleurs_adh cube [|3; 3; 3|])
	in
		printf__sprintf "(%s, %s, %s)" noms.(0) noms.(1) noms.(2)
;;

(*-------------------------------------------------*)

let resoudre_le_cube3x3 cube tester completement =
	(* passer tester = true si le cube 3x3 n'a pas été testé lors de son assemblage, tester = false sinon *)
	(* si tester = true, le test 'fix_cube' est fait au début du traitement du troisième niveau *)
	(* '(y, m) = pos0 x' : le minicube d'indice 'x' est à l'emplacement d'indice 'y' et 'm' est sa matrice *)
	(* de déplacement (telle que 'y=xm') (repère ADH) *)
	(* '(x, m) = pos1 y' : l'emplacement d'indice 'y' est occupé par le minicube d'indice 'x' et 'm' est sa matrice *)
	(* de déplacement inverse (telle que 'x = my') (repère ADH) *)
	
	matr := cube.context1.matrice;
	let encode x = [|x.(0) / 3; x.(1) / 3; x.(2) / 3|]
	and decode x = let f t = if t = 0 then 1 else if t > 0 then 3 else - 3
		in map_vect f x
	in
		let pos0, pos1 =
			let pos mv1 t =
				let x = decode t in
					let p = cube.context1.matrice in
						let m = transpose p /./ (fun_of_mv1 mv1) (x /:/ transpose p) /./ p
						in
							encode (x /:/ m), m
			in
				(fun x -> pos cube.mouvement1.mv1 x),
				(fun x -> pos (inverse cube.mouvement1.mv1) x)
		
		
		and (OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
		and (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) = cube.rotations_faces1 (*tranches externes*)
		in
			
			let op_with_name s =
				let la = [("a", a); ("p", p); ("h", h); ("b", b); ("d", d); ("g", g);
						("a'", a'); ("p'", p'); ("h'", h'); ("b'", b'); ("d'", d'); ("g'", g');
						("a0", a0); ("d0", d0); ("h0", h0); ("a0'", a0'); ("d0'", d0'); ("h0'", h0');]
				in
					assoc s la
			in
				let exec1 str =
					let temp = cube.context1.matrice in
						let listop = op_names_from_string str in
							exe1 (map op_with_name listop);
							let temp1 = cube.context1.matrice in
								mctx := !mctx /./ transpose temp /./ temp1;
				and exec str =
					let listop = op_names_from_string str in
						nbqt := !nbqt + list_length listop;
						if !mctx <> id then
							(
								lo := !lo ^ format_string (decomposition (transpose !mctx));
								mctx := id
							);
						exe1 (map op_with_name listop);
						lo := !lo ^ (format_string str);
						let m = (transpose cube.context1.matrice) /./ !matr in
							print_string (format_string (concat (map (associe m) listop)))
				
				in
					
					let niveau_superieur () =
						(* niveau supérieur *)
						let orienter_le_centre () =
							let n = nqt cube ([|0; 0; 1|]) in
								if n > 0 then (
										print_string "(* ORIENTATION DU CENTRE *)\n";
										lo := !lo ^ "(* ORIENTATION DU CENTRE *)\n";
									);
								if n = 1 then exec "h'"
								else if n = 2 then exec "hh"
								else if n = 3 then exec "h"
						and placer_et_orienter_les_angles () =
							let placer_angle_frontal_haut () =
								let v, _ = pos0 [|1; 0; 1|] in
									match vect v with
										| (1, 0, 1) -> ()
										| (1, 1, 0) -> exec "a'"
										| (1, 0, - 1) -> exec "aa"
										| (1, - 1, 0) -> exec "a"
										| (0, 1, 1) -> exec "d'a'"
										| (- 1, 1, 0) -> exec "h'd'h"
										| (0, 1, - 1) -> exec "da'd'"
										| (0, - 1, 1) -> exec "ga"
										| (- 1, - 1, 0) -> exec "hgh'"
										| (0, - 1, - 1) -> exec "g'ag"
										| (- 1, 0, 1) -> exec "ppbbaa"
										| (- 1, 0, - 1) -> exec "bbaa"
										| _ -> raise Placer_angle_frontal_haut
							and mal_oriente () =
								(snd (pos0 [|1; 0; 1|])).(2) <> [|0; 0; 1|]
							in
								for i = 0 to 3 do
									placer_angle_frontal_haut ();
									if mal_oriente () then
										exec "h'd'ha'";
									exec1 "h0"
								done
						and placer_et_orienter_les_coins () =
							let descendre_coin () =
								let w, m = pos0 [|1; 1; 1|] in
									if (w = [|1; 1; 1|]) && (m = id) then ()
									else
										match vect w with
											| (- 1, 1, 1) -> exec "p'b'p"
											| (- 1, - 1, 1) -> exec "pbbp'"
											| (1, - 1, 1) -> exec "gbg'"
											| (1, 1, 1) -> exec "aba'b'"
											| (- 1, 1, - 1) -> exec "b'"
											| (- 1, - 1, - 1) -> exec "bb"
											| (1, - 1, - 1) -> exec "b"
											| (1, 1, - 1) -> ()
											| _ -> raise Descendre_coin
							and remonter_coin () =
								let (v, m) = pos0 [|1; 1; 1|] in
									if (v = [|1; 1; 1|]) && (m = id) then ()
									else
										let w = m.(2) in match vect w with
												| (1, 0, 0) -> exec "da'd'a"
												| (0, 1, 0) -> exec "a'dad'"
												| (0, 0, - 1) -> exec "ab'a'bbda'd'a"
												| _ -> raise Remonter_coin
							in
								for i = 0 to 3 do
									descendre_coin ();
									remonter_coin ();
									exec1 "h0"
								done;
						in
							if completement then orienter_le_centre ();
							print_string "(* LA CROIX *)\n";
							lo := !lo ^ "(* LA CROIX *)\n";
							placer_et_orienter_les_angles ();
							print_string "(* LES COINS *)\n";
							lo := !lo ^ "(* LES COINS *)\n";
							placer_et_orienter_les_coins ();
					
					and niveau_median () =
						(* niveau médian *)
						let orienter_les_centres_lateraux () =
							print_string "(* ORIENTATION DES CENTRES *)\n";
							lo := !lo ^ "(* ORIENTATION DES CENTRES *)\n";
							let aux () =
								let n = nqt cube [|1; 0; 0|]
								in
									if n = 1 then
										exec "aabba'bbaa"
									else if n = 2 then
										exec "aabbaabbaa"
									else if n = 3 then
										exec "aabbabbaa"
							in
								(aux (); exec1 "h0"; aux (); exec1 "h0"; aux (); exec1 "h0"; aux (); exec1 "h0");
								print_string "(* ORIENTATION DES CENTRES TERMINÉE *)\n";
								lo := !lo ^ "(* ORIENTATION DES CENTRES TERMINÉE *)\n";
						and placer_angle_frontal_droit () =
							let descendre_angle () =
								let aux () = exec "bab'a'b'd'bd" in
									let x, _ = pos0 [|1; 1; 0|] in
										match vect x with
											| (1, 1, 0) -> aux ()
											| (- 1, 1, 0) -> exec1 "h0"; aux (); exec1 "h0'"
											| (- 1, - 1, 0) -> exec1 "h0"; exec1 "h0"; aux (); exec1 "h0"; exec1 "h0";
											| (1, - 1, 0) -> exec1 "h0'"; aux (); exec1 "h0"
											| _ -> ()
							and remonter_angle () =
								let aux_r () = exec "b'd'bdbab'a'"
								and aux_l () = exec "bab'a'b'd'bd"
								in
									let x, m = pos0 [|1; 1; 0|] in
										if m.(0) <> [|0; 0; - 1|] then
											match vect x with
												| (1, 0, - 1) -> aux_r ()
												| (0, - 1, - 1) -> exec "b"; aux_r ()
												| (- 1, 0, - 1) -> exec "bb"; aux_r ()
												| (0, 1, - 1) -> exec "b'"; aux_r ()
												| _ -> raise Remonter_angle
										else
											match vect x with
												| (1, 0, - 1) -> exec "b"; aux_l ()
												| (0, - 1, - 1) -> exec "bb"; aux_l ()
												| (- 1, 0, - 1) -> exec "b'"; aux_l ()
												| (0, 1, - 1) -> aux_l ()
												| _ -> raise Remonter_angle
							in
								let x, m = pos0 [|1; 1; 0|] in
									if (x = [|1; 1; 0|]) && (m = id) then ()
									else (
											descendre_angle ();
											remonter_angle ()
										)
						in
							if completement then orienter_les_centres_lateraux ();
							for i = 0 to 3 do
								placer_angle_frontal_droit ();
								exec1 "h0"
							done
					
					and niveau_inferieur () =
						(* niveau inférieur *)
						let orienter_les_angles () =
							let est_mal_oriente angle =
								let (_, m) = pos1 angle in
									m.(2) <> [|0; 0; 1|]
							in
								let v = map_vect est_mal_oriente
									[|[|1; 0; - 1|]; [|0; - 1; - 1|]; [|- 1; 0; - 1|]; [|0; 1; - 1|]|]
								in match (v.(0), v.(1), v.(2), v.(3)) with
										| (false, false, false, false) -> ()
										| (true, true, true, true) -> exec "dbab'a'd'bdaba'b'd'"
										
										| (false, false, true, true) -> exec1 "h0"; exec "dbab'a'd'"
										| (true, false, false, true) -> exec "dbab'a'd'"
										| (true, true, false, false) -> exec1 "h0'"; exec "dbab'a'd'"
										| (false, true, true, false) -> exec1 "h0"; exec1 "h0"; exec "dbab'a'd'"
										
										| (false, _, false, _) -> exec "daba'b'd'"
										| (_, false, _, false) -> exec1 "h0"; exec "daba'b'd'"
										| _ -> raise Orienter_les_angles
						
						and placer_les_angles () =
							let permuter () =
								(* laisse fixe l'angle arrière et permute circulairement les autres *)
								(* dans le sens direct vu d'en bas *)
								exec "dbbd'b'db'd'"
							and permuter' () =
								(* laisse fixe l'angle arrière et permute circulairement les autres *)
								(* dans le sens indirect vu d'en bas *)
								exec "dbd'bdbbd'"
							in
								let chercher_un_angle_bien_place () =
									let i = ref 0 in
										while !i < 4 && fst (pos0 [|- 1; 0; - 1|]) <> [|- 1; 0; - 1|] do
											exec1 "h0";
											incr i
										done;
										!i
								in
								(* la méthode utilisée suppose ici que la permutation des angles soit paire *)
									if completement then (* maintenir la somme des rotations des milieux nulle modulo 4, *)
									(*ce qui implique une permutation paire des coins donc des angles du cube 3x3 *)
										(let n = nqt cube ([|0; 0; - 1|]) in
												if n = 1 then exec "b'" else if n = 3 then exec "b" else if n = 2 then exec "bb"
										)
									else (* maintenir une permutation paire des angles donc des coins *)
										(
											if sign_angles cube.mouvement1.mv1 = - 1 then exec "b"
											else ()
										)
									;
									let j = chercher_un_angle_bien_place () in
										if j = 4 (* aucun angle bien placé *) then
											(
												permuter ();
												let _ = chercher_un_angle_bien_place () in ()
											)
										else ();
										let v, _ = pos0 [|1; 0; - 1|] in match vect v with
												| (0, - 1, - 1) -> permuter ()
												| (0, 1, - 1) -> permuter' ()
												| (1, 0, - 1) -> ()
												| _ -> raise Placer_les_angles
						
						and placer_les_coins () =
							(* à ce stade la permutation des coins devrait être paire *)
							let permuter () =
								(* laisse fixe le coin frontal droit et permute circulairement 
          						 les autres dans le sens direct vu d'en bas *)
								exec "bab'p'ba'b'p"
							and permuter' () =
								(* laisse fixe le coin frontal droit et permute circulairement 
                       les autres dans le sens indirect vu d'en bas *)
								exec "p'bab'pba'b'"
							in
								let chercher_un_coin_bien_place () =
									let i = ref 0 in
										while !i < 4 && fst (pos0 [|1; 1; - 1|]) <> [|1; 1; - 1|] do
											exec1 "h0";
											incr i
										done;
										!i
								in
									let j = chercher_un_coin_bien_place () in
										
										if j = 4 (* aucun coin bien placé *) then (
												permuter ();
												let _ = chercher_un_coin_bien_place () in ()
											)
										else ();
										let v, _ = pos0 [|- 1; - 1; - 1|] in match vect v with
												| (1, - 1, - 1) -> permuter ()
												| (- 1, 1, - 1) -> permuter' ()
												| (- 1, - 1, - 1) -> ()
												| _ -> raise Placer_les_coins
						
						and orienter_les_coins () =
							let faire_tourner () =
								(* fait tourner les coins frontaux inférieurs sur eux-mêmes: 
                       le coin gauche dans le sens direct, le coin droit en sens inverse *)
								exec "p'b'pb'p'bbp";
								exec "aba'babba'"
							and
							faire_tourner' () =
								(* fait tourner les coins frontaux inférieurs sur eux-mêmes:
                       le coin droit dans le sens direct, le coin gauche en sens inverse *)
								exec "abba'b'ab'a'";
								exec "p'bbpbp'bp"
							in
								let orienter_frontal_inferieur_droit () =
									let _, m = pos0 [|1; 1; - 1|] in
										let v = m.(2) in
											match vect v with
												| (0, 0, 1) -> ()
												| (- 1, 0, 0) -> faire_tourner' ()
												| (0, - 1, 0) -> faire_tourner ()
												| _ -> raise Orienter_les_coins
								in
									for i = 0 to 2 do
										orienter_frontal_inferieur_droit ();
										exec1 "h0'"
									done
						in
							print_string "(* LA CROIX *)\n";
							lo := !lo ^ "(* LA CROIX *)\n";
							orienter_les_angles ();
							placer_les_angles ();
							print_string "(* LES COINS *)\n";
							lo := !lo ^ "(* LES COINS *)\n";
							placer_les_coins ();
							orienter_les_coins ();
					in
						if completement then
						(* les centres formant les milieux du cube 3x3 sont censés être à leurs places après appel de la fonction 'arranger les centres' *)
						(* éviter maintenant la rotation de ces centres *)
							cube.mouvement1.mv1 <- map (fun (x, m) -> x, if not (est_centre x || est_angle x || est_coin x) then id else m) cube.mouvement1.mv1;
						let ctx = cube.context1.matrice in
							try
								(
									let n = !nbqt in
										(
											printf__printf "\n(* NIVEAU SUPÉRIEUR *)\n";
											lo := !lo ^ printf__sprintf "\n(* NIVEAU SUPÉRIEUR *)\n";
											niveau_superieur ();
											(
												printf__printf "(* FIN DE NIVEAU SUPÉRIEUR : %s *)\n" (nbqdt (!nbqt - n));
												lo := !lo ^ printf__sprintf "(* FIN DE NIVEAU SUPÉRIEUR : %s *)\n" (nbqdt (!nbqt - n));
												print_newline ()
											)
										)
								);
								(
									let n = !nbqt in
										(
											printf__printf "(* NIVEAU MÉDIAN *)\n";
											lo := !lo ^ printf__sprintf "\n(* NIVEAU MÉDIAN *)\n";
											niveau_median ();
											(
												printf__printf "(* FIN DE NIVEAU MÉDIAN : %s *)\n" (nbqdt (!nbqt - n));
												lo := !lo ^ printf__sprintf "(* FIN DE NIVEAU MÉDIAN : %s *)\n" (nbqdt (!nbqt - n));
												print_newline ()
											)
										)
								);
								if tester then
								(* test à faire si le cube 3x3 n'a pas été testé lors de l'assemblage de ses angles et de ses milieux *)
									(
										let n = !nbqt in
											(
												printf__printf "(* TRAITEMENT DES PARITÉS *)\n";
												lo := !lo ^ printf__sprintf "\n(* TRAITEMENT DES PARITÉS *)\n";
												fix_cube cube;
												(
													printf__printf "(* FIN DE TRAITEMENT DES PARITÉS : %s *)\n" (nbqdt (!nbqt - n));
													lo := !lo ^ printf__sprintf "(* FIN DE TRAITEMENT DES PARITÉS : %s *)\n" (nbqdt (!nbqt - n));
													print_newline ()
												)
											)
									);
								(
									let n = !nbqt in
										(
											printf__printf "(* NIVEAU INFÉRIEUR *)\n";
											lo := !lo ^ printf__sprintf "\n(* NIVEAU INFÉRIEUR *)\n";
											niveau_inferieur ();
											(* pour ramener le cube à sa position initiale *)
											(
												let s = decomposition (transpose (!mctx /./ transpose cube.context1.matrice /./ !matr)) in
													if s <> "" then
														(
															lo := !lo ^ format_string s;
														);
											);
											(
												printf__printf "(* FIN DE NIVEAU INFÉRIEUR : %s *)" (nbqdt (!nbqt - n));
												lo := !lo ^ printf__sprintf "(* FIN DE NIVEAU INFÉRIEUR : %s *)\n" (nbqdt (!nbqt - n));
												print_newline ()
											)
										)
								);
								cube.context1.matrice <- ctx;
								dessine_cube cube;
								!nbqt
							with
								| Orienter_les_coins ->
											print_string "orienter_les_coins\n"; !nbqt
								| Placer_les_coins ->
											print_string "placer_les_coins\n"; !nbqt
								| Placer_les_angles ->
											print_string "placer_les_angles\n"; !nbqt
								| Orienter_les_angles ->
											print_string "orienter_les_angles\n"; !nbqt
;;

(*-------------------------------------------------------------------------------------------*)
(* 11 : INITIALISATION D'UN CUBE 4x4 *)
(*-------------------------------------------------------------------------------------------*)

(*- mise en place des mouvements globaux et des mouvements élémentaires de Rubik -*)

let affiche_mvt repere context mv1 = affiche_mouvement repere.plan context mv1;;

let nouveau_cube mouvement context repere anime =
	let dessine () = if !anime then affiche_mvt repere context mouvement.mv1
	in
		let rotations_facesi () =
			let fct x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub t;
					dessine ()
			and fct' x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub' t;
					dessine ()
			in
				let (a, d, h) = vect (map_vect fct id)
				and (a', d', h') = vect (map_vect fct' id)
				and (p, g, b) = vect (map_vect fct idm)
				and (p', g', b') = vect (map_vect fct' idm)
				in (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b'))
		and rotations_faces () =
			let fct x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub3 t;
					dessine ()
			and fct' x () =
				let t = x /:/ transpose context.matrice in
					mouvement.mv1 <- mouvement.mv1 /*/ rub3' t;
					dessine ()
			in
				let (a, d, h) = vect (map_vect fct id)
				and (a', d', h') = vect (map_vect fct' id)
				and (p, g, b) = vect (map_vect fct idm)
				and (p', g', b') = vect (map_vect fct' idm)
				in (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b'))
		and rotations_cube () =
			let rotate pp () =
				context.matrice <- context.matrice /./ pp;
				dessine ()
			in
				let (a, d, h) = vect (map_vect rotate (map_vect rot id))
				and (a', d', h') = vect (map_vect rotate (map_vect rot' id))
				in
					(OPS (a, d, h), OPS (a', d', h'))
		in
			{
				anime1 = anime;
				mouvement1 = mouvement;
				mvi = {mv1 = mv1_of_fun (fun x -> if est_axe x then id else matrice_nulle)};
				context1 = context;
				repere1 = repere;
				rotations_cube1 = rotations_cube ();
				rotations_faces1 = rotations_faces ();
				rotations_faces1i = rotations_facesi ();
				boutons1 = make_vect 1 {titre = ""; orx = 0; ory = 0; largeur = 0;
					hauteur = 0; couleur = BLANC; action = (fun () -> ()); bas = false}
			}
;;

(*---------------------------------------------------------------------------------------------*)
(* 12 : RÉDUCTION D'UN MOUVEMENT 4x4 à un MOUVEMENT 3x3 : regroupement des centres par couleur *)
(*---------------------------------------------------------------------------------------------*)

let est_dans_face cube c = (* 'c' est un indice de centre *)
	let normale c = map_vect (fun x -> x / 3) c
	in
		normale c = normale (sur cube.mouvement1.mv1 c)
;;

let est_en_face cube c = (* 'c' est un indice de centre *)
	let normale c = map_vect (fun x -> x / 3) c
	in
		normale c = map_vect (fun x -> - x) (normale (sur cube.mouvement1.mv1 c))
;;

let faire_tourner1 cube c = (* face contenant le centre d'indice 'c' et face destination opposées *)
	(* positionnement de la face contenant le centre en bas, avec le centre en avant-gauche, et de la face destination en haut *)
	let p = cube.context1.matrice
	and normale c = map_vect (fun x -> x / 3) c
	in
		let normale_face = normale c /:/ p
		and position_centre = sur cube.mouvement1.mv1 c /:/ p
		in
			let mat = hd (select (fun m -> normale_face /:/ m = [|0; 0; 1|] && position_centre /:/ m = [|1; - 1; - 3|]) groupe_du_cube)
			in decomposition (transpose mat)
;;

let faire_tourner2 cube c = (* face contenant le centre d'indice 'c'  et face destination adjacentes *)
	(* positionnement de la face contenant le centre à l'avant et de la face destination en haut *)
	let p = cube.context1.matrice
	and normale c = map_vect (fun x -> x / 3) c
	in
		let normale_face = normale c /:/ p
		and position_centre = sur cube.mouvement1.mv1 c /:/ p
		in
			let mat = hd (select (fun m -> normale_face /:/ m = [|0; 0; 1|] && (position_centre /:/ m).(0) = 3) groupe_du_cube)
			in decomposition (transpose mat)
;;

let place_centre cube c = (* c est l'indice d'un centre *)
	let (_, (exec1, exec)) = operations cube
	in
		let m1 () = exec "gi'gi'hgigi" and m2 () = exec "gi'hgi" in
			if not (est_dans_face cube c) then
				(
					if est_en_face cube c
					then (* le centre 'c' est sur la face opposée à la face destination *)
						((* positionnement de la face destination en haut et de 'c' en [|1;-1;-3|]*)
							(let str = faire_tourner1 cube c in exec1 str);
							(* positionnement par quart(s) de tour de la face haute *)
							if est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|1; 1; 3|] /:/ transpose cube.context1.matrice)) then
								if not (est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|- 1; 1; 3|] /:/ transpose cube.context1.matrice))) then
									exec "h"
								else if not (est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|1; - 1; 3|] /:/ transpose cube.context1.matrice))) then
									exec "h'"
								else exec "hh";
								m1 ()
						)
					else (* le centre 'c' est sur une face adjacente à la face destination *)
						(
							(let st = ref "" in
									(let str = faire_tourner2 cube c in exec1 str);
									(* positionnement par quart(s) de tour de la face antérieure *)
									(let c1 = sur cube.mouvement1.mv1 c /:/ cube.context1.matrice in
											match vect c1 with
												| (3, - 1, - 1) -> st := !st ^ "a"
												| (3, 1, 1) -> st := !st ^ "a'"
												| (3, 1, - 1) -> st := !st ^ "aa"
												| _ -> ()
									);
									(* positionnement par quart(s) de tour de la face haute *)
									if est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|1; 1; 3|] /:/ transpose cube.context1.matrice)) then
										(if not (est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|- 1; 1; 3|] /:/ transpose cube.context1.matrice))) then
												st := !st ^ "h"
											else if not (est_dans_face cube (sur (inverse cube.mouvement1.mv1) ([|1; - 1; 3|] /:/ transpose cube.context1.matrice))) then
												st := !st ^ "h'"
											else
												st := !st ^ "hh"
										);
									exec !st
							);
							m2 ()
						);
				);
;;

let regrouper_les_centres cube =
	let (_, (_, exec)) = operations cube in
		if sign angles (sur cube.mouvement1.mv1) = - 1 then
		(* parité d'orientation 'OLL' *)
			exec "hi"; (* par exemple *)
		let rec aux l = match l with
				| t :: r -> place_centre cube t; aux r
				| [] -> ()
		in
			let p = cube.context1.matrice
			in
				aux (select est_centre indices);
				((* ramener le cube à sa position initiale *)
					let s = decomposition (transpose p /./ cube.context1.matrice) in
						if s <> "" then
							(
								lo := !lo ^ printf__sprintf "exec \"%s\";;\n" s;
							);
				);
				cube.context1.matrice <- p;
				dessine_cube cube;
;;


(*-------------------------------------------------------------------------------------------*)
(* 13 : RÉDUCTION D'UN MOUVEMENT 4x4 à un MOUVEMENT 3x3 : arrangement des centres            *)
(*-------------------------------------------------------------------------------------------*)

(*-- arranger les centres : seulement en vue d'une résolution complète du cube --*)

(* fait en sorte que chaque centre soit à sa place dans sa face. *)
(* sachant qu'on a déjà amené chaque centre dans la face ayant même *)
(* couleur que lui à l'aide de la fonction 'regrouper_les_centres' *)

exception Arranger_les_centres;;

let arranger_les_centres cube =
	let (_, (exec1, exec)) = operations cube
	and pos0, pos1 =
		let pos cube mv1 x =
			let pp = cube.context1.matrice in
				let m = transpose pp /./ (fun_of_mv1 mv1) (x /:/ transpose pp) /./ pp
				in
					x /:/ m, m
		in
			(fun cube x -> pos cube cube.mouvement1.mv1 x),
			(fun cube x -> pos cube (inverse cube.mouvement1.mv1) x)
	and centres_dans_face cube i =
		let encode x = [|x.(0) / 3; x.(1) / 3; x.(2) / 3|] in
			let est_dans_face cube i x = i /:/ transpose cube.context1.matrice = encode x
			in select (est_dans_face cube i) centres
	in
		let ah () =
			let ll = "aibiai'bi'h'biaibi'ai'h"
			and ll' = "h'aibiai'bi'hbiaibi'ai'"
			in
				let m () = exec ll
				and m' () = exec ll'
				in (* faire en sorte d'abord que la permutations des centres hauts soit paire *)
					if sign (centres_dans_face cube [|0; 0; 1|]) (sur cube.mouvement1.mv1) = - 1 then exec "h";
					let faire_le_tour () =
						let i = ref 0 in
							while !i < 4 && fst (pos0 cube [|- 1; 1; 3|]) <> [|- 1; 1; 3|] do
								exec1 "h0";
								incr i
							done;
							if !i < 4 then (
									let x = fst (pos0 cube [|- 1; - 1; 3|]) in
										if x = [|1; 1; 3|] then
											(exec "pibi"; m' (); exec "bi'pi'";)
										else if x = [|1; - 1; 3|] then
											(exec "pibi"; m (); exec "bi'pi'";);
										while !i < 4 do
											exec1 "h0";
											incr i
										done;
										false
								)
							else (
									exec "pibi"; m' (); exec "bi'pi'";
									true
								)
					in
						if (faire_le_tour () && faire_le_tour ()) then
							raise Arranger_les_centres
		in
			ah (); exec1 "d0d0"; ah (); exec1 "d0'"; ah (); exec1 "a0";
			ah (); exec1 "a0"; ah (); exec1 "a0"; ah (); exec1 "a0d0'"
;;

(*-- fin de arranger les centres en vue d'une résolution complète du cube --*)


(*-------------------------------------------------------------------------------------------*)
(* 14 : RÉDUCTION D'UN MOUVEMENT 4x4 à un MOUVEMENT 3x3 : appariement des angles *)
(*-------------------------------------------------------------------------------------------*)

let deplace_angle cube x =
	(* déplacement de l'arête 'x' vers [|3;-3;1|] *)
	let (_, (exec1, _)) = operations cube
	in
		exec1 (decomposition (hd (select (fun m -> [|3; - 3; 1|] /:/ m = x) groupe_du_cube)))
;;

let apparier_angle_ag cube y =
	(* déplacement de l'arête jumelle 'y' vers [|3;3;1|] *)
	let (_, (_, exec)) = operations cube in
		(match vect y with
				| 3, - 1, 3 -> exec "g'ag"
				| 3, 1, 3 -> exec "h'd'"
				| - 1, - 3, 3 -> exec "h'g'ag"
				| 1, - 3, 3 -> exec "h'h'd'"
				| - 1, 3, 3 -> exec "d'"
				| 1, 3, 3 -> exec "a'ha"
				| - 3, - 1, 3 -> exec "hd'"
				| - 3, 1, 3 -> exec "p'd'd'"
				
				| 3, - 1, - 3 -> exec "bd"
				| 3, 1, - 3 -> exec "g'a'g"
				| - 1, - 3, - 3 -> exec "g'aag"
				| 1, - 3, - 3 -> exec "g'agh'd'"
				| - 1, 3, - 3 -> exec "b'g'a'g"
				| 1, 3, - 3 -> exec "d"
				| - 3, - 1, - 3 -> exec "bbg'a'g"
				| - 3, 1, - 3 -> exec "b'd"
				
				| - 3, - 3, - 1 -> exec "p'hd'"
				| - 3, - 3, 1 -> exec "ppdd"
				| - 3, 3, - 1 -> exec "dd"
				| - 3, 3, 1 -> exec "phd'"
				| 3, 3, - 1 -> exec "da'ha"
				
				| _ -> ()
		);
		(* appariement *)
		exec "bida'hd'abi'";
;;

(* tant que nécessaire, déplacer un angle non apparié vers [|3;-3;1|], chercher son jumeau et apparier *)
let apparier_les_angles cube =
	let p = cube.context1.matrice
	in
		let jumeau a =
			let f x = if x = 1 then - 1 else if x = - 1 then 1 else x in map_vect f a
		in
			let est_a_apparier cube x =
				(sur cube.mouvement1.mv1) (jumeau x) <> jumeau ((sur cube.mouvement1.mv1) x)
			in
				let l = ref (select (fun x -> est_angle x && (x.(0) = 1 || x.(1) = 1 || x.(2) = 1)) (map fst e))
				in
					while !l <> [] do
						let t = hd !l in
							if est_a_apparier cube t then
								(
									let t1 = (sur cube.mouvement1.mv1 t) /:/ cube.context1.matrice in
										deplace_angle cube (t1);
										apparier_angle_ag cube (sur cube.mouvement1.mv1 (jumeau t) /:/ cube.context1.matrice);
								);
							l := tl !l
					done;
					(* parité de permutation 'PLL' *)
					(if sign_angles cube.mouvement1.mv1 <> sign coins (sur cube.mouvement1.mv1) then
							let (_, (_, exec)) = operations cube in
								exec "didihhdidihihihhdidihihi"
					);
					((* ramener le cube à sa position initiale *)
						let s = decomposition (transpose p /./ cube.context1.matrice) in
							if s <> "" then
								(
									lo := !lo ^ printf__sprintf "exec \"%s\";;\n" s;
								);
					);
					cube.context1.matrice <- p;
					dessine_cube cube;
;;

(*-- fin de apparier les angles --*)

(*- fin de étapes pour la résolution par niveaux du Rubik's cube 4x4 -*)


(*-------------------------------------------------------------------------------------------*)
(* 15 : RÉSOLUTION DU CUBE 4x4 *)
(*-------------------------------------------------------------------------------------------*)

(* Résolution du cube 4x4 ne prenant en compte que les couleurs des faces *)
let resoudre_le_cube_4x4 cube =
	let mat = cube.context1.matrice
	and noms_couleurs = noms_couleurs_coin_adh cube
	in
		print_string "include \"exemples/Caml Light/Rubik/interfaces/interface4.ml\";;\n\n";
		printf__printf "(* coin Antérieur Droit Haut : %s *)\n\n" noms_couleurs;
		
		printf__printf "let mv = lire_mouv \"mouv4444\";;\n";
		printf__printf "let ctx = context_adh %s mv;;\n" noms_couleurs;
		printf__printf "graphics__open_graph \" 666x800\";;\n\n";
		
		print_string "(* RÉSOLUTION SIMPLE SANS ROTATIONS GLOBALES *)\n\n";
		printf__printf "(* %s *)\n\n" (couleurs_faces_adh cube);
		
		printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
		printf__printf "cube.mouvement1.mv1 <- mv;;\n";
		printf__printf "dessine_cube ctx mv;;\n";
		
		cube.context1.matrice <- mat;
		dessine_cube cube;
		
		print_string "\n(* RÉDUCTION À UN CUBE 3 x 3 : *)\n";
		nbqt := 0;
		lo := "";
		matr := cube.context1.matrice;
		mctx := id;
		
		lo := !lo ^ "\n(* RÉDUCTION À UN CUBE 3 x 3 : *)\n";
		
		(let n = !nbqt in
				print_string "\n(* DÉBUT DE REGROUPER LES CENTRES *)\n";
				lo := !lo ^ "\n(* DÉBUT DE REGROUPER LES CENTRES *)\n";
				regrouper_les_centres cube; (* regroupement par couleurs des centres sur une même face *)
				printf__printf "(* FIN DE REGROUPER LES CENTRES : %s *)\n" (nbqdt (!nbqt - n));
				lo := !lo ^ printf__sprintf "(* FIN DE REGROUPER LES CENTRES : %s *)\n" (nbqdt (!nbqt - n));
		);
		
		print_string "\n(* DÉBUT DE APPARIER LES ANGLES : *)\n";
		lo := !lo ^ "\n(* DÉBUT DE APPARIER LES ANGLES *)\n";
		
		(let n = !nbqt in
				apparier_les_angles cube; (* réapparier les demi-angles *)
				printf__printf "(* FIN DE APPARIER LES ANGLES : %s *)\n" (nbqdt (!nbqt - n));
				lo := !lo ^ printf__sprintf "(* FIN DE APPARIER LES ANGLES : %s *)\n" (nbqdt (!nbqt - n));
		);
		
		printf__printf "\n(* FIN DE RÉDUCTION À UN CUBE 3 x 3 : %s *)\n" (nbqdt !nbqt);
		lo := !lo ^ printf__sprintf "\n(* FIN DE RÉDUCTION À UN CUBE 3 x 3 : %s *)\n" (nbqdt !nbqt);
		
		print_string "\n\n(* RÉSOLUTION DU CUBE 3 x 3 SOUS-JACENT *)\n";
		lo := !lo ^ "\n\n(* RÉSOLUTION DU CUBE 3 x 3 SOUS-JACENT *)\n";
		(* résolution du cube 3x3 sans tests, sans rotation des centres *)
		let n = resoudre_le_cube3x3 cube false false
		in
			printf__printf "\n(* RÉSOLUTION SIMPLE DU CUBE 4 x 4 en %s *)\n" (nbqdt !nbqt);
			lo := !lo ^ printf__sprintf "\n(* RÉSOLUTION SIMPLE DU CUBE 4 x 4 en %s *)\n" (nbqdt !nbqt);
			
			print_string "\n(*-----------------------------------------------------------*)\n";
			print_string "\n(* RÉSOLUTION SIMPLE AVEC ROTATIONS GLOBALES *)\n\n";
			printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
			printf__printf "cube.mouvement1.mv1 <- mv;;\n";
			printf__printf "dessine_cube ctx mv;;\n\n";
			printf__printf "%s\n" !lo;
			n
;;

(* résolution du cube 4x4 avec retour à la configuration usine *)

let resoudre_completement_le_cube_4x4 cube =
	let mat = cube.context1.matrice
	and noms_couleurs = noms_couleurs_coin_adh cube
	in
		print_string "include \"exemples/Caml Light/Rubik/interfaces/interface4.ml\";;\n\n";
		printf__printf "(* coin Antérieur Droit Haut : %s *)\n\n" noms_couleurs;
		
		printf__printf "let mv = lire_mouv \"mouv4444\";;\n";
		printf__printf "let ctx = context_adh %s mv;;\n" noms_couleurs;
		printf__printf "graphics__open_graph \" 666x800\";;\n\n";
		print_string "(* RÉSOLUTION COMPLÈTE SANS ROTATIONS GLOBALES *)\n\n";
		printf__printf "(* %s *)\n\n" (couleurs_faces_adh cube);
		
		printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
		printf__printf "cube.mouvement1.mv1 <- mv;;\n";
		printf__printf "dessine_cube ctx mv;;\n\n";
		
		cube.context1.matrice <- mat;
		dessine_cube cube;
		
		print_string "\n(* RÉDUCTION À UN CUBE 3 x 3 : *)\n";
		nbqt := 0;
		lo := "";
		matr := cube.context1.matrice;
		mctx := id;
		
		lo := !lo ^ "\n(* RÉDUCTION À UN CUBE 3 x 3 : *)\n";
		
		(let n = !nbqt in
				print_string "\n(* DÉBUT DE REGROUPER LES CENTRES *)\n";
				lo := !lo ^ "\n(* DÉBUT DE REGROUPER LES CENTRES *)\n";
				regrouper_les_centres cube; (* regroupement par couleurs des centres sur une même face *)
				print_string "(* FIN DE REGROUPER LES CENTRES *)\n";
				lo := !lo ^ printf__sprintf "(* FIN DE REGROUPER LES CENTRES : %s *)\n" (nbqdt (!nbqt - n));
		);
		
		print_string "\n(* DÉBUT DE APPARIER LES ANGLES *)\n";
		lo := !lo ^ "\n(* DÉBUT DE APPARIER LES ANGLES *)\n";
		
		(let n = !nbqt in
				apparier_les_angles cube; (* réapparier les demi-angles *)
				printf__printf "(* FIN DE APPARIER LES ANGLES : %s *)\n" (nbqdt (!nbqt - n));
				lo := !lo ^ printf__sprintf "(* FIN DE APPARIER LES ANGLES : %s *)\n" (nbqdt (!nbqt - n));
		);
		
		(let n = !nbqt in
				print_string "\n(* DÉBUT DE ARRANGER LES CENTRES *)\n";
				lo := !lo ^ "\n(* DÉBUT DE ARRANGER LES CENTRES *)\n";
				arranger_les_centres cube; (* remettre chaque centre à sa place *)
				printf__printf "(* FIN DE ARRANGER LES CENTRES : %d *)\n" (!nbqt - n);
				lo := !lo ^ printf__sprintf "(* FIN DE ARRANGER LES CENTRES  : %s *)\n" (nbqdt (!nbqt - n));
		);
		
		printf__printf "\n(* FIN DE RÉDUCTION À UN CUBE 3 x 3 : %s *)\n" (nbqdt !nbqt);
		lo := !lo ^ printf__sprintf "\n(* FIN DE RÉDUCTION À UN CUBE 3 x 3 : %s *)\n" (nbqdt !nbqt);
		print_string "\n\n(* RÉSOLUTION DU CUBE 3 x 3 SOUS-JACENT *)\n";
		lo := !lo ^ "\n\n(* RÉSOLUTION DU CUBE 3 x 3 SOUS-JACENT *)\n";
		
		let n = resoudre_le_cube3x3 cube false true (* résolution du cube 3x3 sans tests, avec rotation des centres *)
		in
			printf__printf "\n(* RÉSOLUTION COMPLÈTE DU CUBE 4 x 4 en %s *)\n" (nbqdt !nbqt);
			lo := !lo ^ printf__sprintf "\n(* RÉSOLUTION COMPLÈTE DU CUBE 4 x 4 en %s *)\n" (nbqdt !nbqt);
			
			print_string "\n(*-----------------------------------------------------------*)\n";
			print_string "\n(* RÉSOLUTION COMPLÈTE AVEC ROTATIONS GLOBALES *)\n\n";
			printf__printf "cube.context1.matrice <- ctx.matrice;;\n";
			printf__printf "cube.mouvement1.mv1 <- mv;;\n";
			printf__printf "dessine_cube ctx mv;;\n\n";
			printf__printf "%s\n" !lo;
			n
;;

(* test d'appartenance d'un mouvement au sous-groupe R *)
(* fondé sur la résolution précédente *)
let est_rubik m =
	let cube = nouveau_cube {mv1 = m} {matrice = id} {plan = (0, 0, 0, 0)} (ref false)
	in
		try
			regrouper_les_centres cube;
			apparier_les_angles cube;
			arranger_les_centres cube;
			let _ = resoudre_le_cube3x3 cube false true
			in
				cube.mouvement1.mv1 = e
		with Failure s -> false
;;

(*-------------------------------------------------------------------------------------------*)
(* 16 : BOUTONS *)
(*-------------------------------------------------------------------------------------------*)

let bouton titre orx ory largeur hauteur couleur action =
	{titre = titre; orx = orx; ory = ory; hauteur = hauteur; largeur = largeur;
		couleur = couleur; action = action; bas = false}
;;

let inverse_bouton b =
	b.bas <- true;
	graphics__set_color graphics__black;
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__white;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let dessine_bouton b =
	b.bas <- false;
	graphics__set_color (couleur_rvb_de_couleur (b.couleur));
	graphics__fill_rect b.orx b.ory b.largeur b.hauteur;
	let (x, y) = graphics__text_size b.titre in
		graphics__moveto (b.orx + (b.largeur - x) / 2) (b.ory + (b.hauteur - y) / 2);
		graphics__set_color graphics__black;
		graphics__draw_string b.titre;
		let p = make_vect 4 (0, 0) in
			p.(0) <- (b.orx, b.ory);
			p.(1) <- (b.orx + b.largeur, b.ory);
			p.(2) <- (b.orx + b.largeur, b.ory + b.hauteur);
			p.(3) <- (b.orx, b.ory + b.hauteur);
			graphics__set_color graphics__black;
			drawPoly p
;;

let set_action bouton action =
	bouton.action <- action
;;

let gestion_bouton bouton mouse_down mousex mousey =
	if bouton.orx < mousex && mousex < bouton.orx + bouton.largeur
		&& bouton.ory < mousey && mousey < bouton.ory + bouton.hauteur then (
			if mouse_down then (
					if not bouton.bas then (
							inverse_bouton bouton;
						)
				)
			else (
					if bouton.bas then (
							dessine_bouton bouton;
							bouton.action ()
						)
				)
		)
	else (
			if bouton.bas then (
					dessine_bouton bouton
				)
		)
;;

(*-------------------------------------------------------------------------------------------*)
(* 17 : RANGÉE DE BOUTONS POUR LA MANIPULATION DU CUBE 4x4 *)
(*-------------------------------------------------------------------------------------------*)

(* Gestion par boutons des mouvements globaux et des mouvements de Rubik *)
(* Pour fenêtre de largeur 666 et hauteur 800 *)
(* Cube avec en bas de fenêtre une rangée de 18 boutons de largeur 37 : 18 x 37 = 666 *)
(* et une rangée de 12 boutons justifiée à droite au dessus *)

(* Fenêtre de largeur 666 et hauteur 800 : origine  au centre (333,400), unités : 20,20 *)

(* Rangée de 18 boutons en bas de fenêtre pour les mouvements globaux et les rotations des faces externes *)

let dessine_boutons1 cube =
	let couleur_titre titre =
		let face titre = match titre with
				| "A" | "A'" | "ai" | "ai'" | "a" | "a'" -> [|1; 0; 0|]
				| "D" | "D'" | "di" | "di'" | "d" | "d'" -> [|0; 1; 0|]
				| "H" | "H'" | "hi" | "hi'" | "h" | "h'" -> [|0; 0; 1|]
				| "pi" | "pi'" -> [|- 1; 0; 0|]
				| "gi" | "gi'" -> [|0; - 1; 0|]
				| "bi" | "bi'" -> [|0; 0; - 1|]
				| "p" | "p'" -> [|- 1; 0; 0|]
				| "g" | "g'" -> [|0; - 1; 0|]
				| "b" | "b'" -> [|0; 0; - 1|]
				| _ -> print_string titre; failwith "face"
		in
			couleur_de_face ((face titre) /:/ transpose cube.context1.matrice)
	in
		for i = 0 to vect_length cube.boutons1 - 1 do
			cube.boutons1.(i).couleur <- couleur_titre cube.boutons1.(i).titre;
			dessine_bouton cube.boutons1.(i)
		done
;;

let cree_boutons1 cube =
	let titres =
		[|"A"; "A'"; "H"; "H'"; "D"; "D'";
			"a"; "a'"; "h"; "h'"; "d"; "d'";
			"p"; "p'"; "b"; "b'"; "g"; "g'"
		|]
	in
		let set_actions boutons =
			let (OPS (a0, d0, h0), OPS (a0', d0', h0')) = cube.rotations_cube1
			and (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) =
				cube.rotations_faces1
			in
				let v = [|a0; a0'; h0; h0'; d0; d0'; a; a'; h; h'; d; d'; p; p'; b; b'; g; g'|]
				and op_names = [|"a0"; "a0'"; "h0"; "h0'"; "d0"; "d0'"; "a"; "a'"; "h"; "h'"; "d"; "d'"; "p"; "p'"; "b"; "b'"; "g"; "g'"|]
				in
					for i = 0 to 5 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube;
											dessine_boutons1 cube;
											flush std_out)
					done;
					for i = 6 to vect_length v - 1 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i));
											dessine_cube cube; flush std_out)
					done
		in
			let n = vect_length titres in
				let boutons = make_vect n (bouton "" 0 0 0 0 BLANC (fun () -> ())) in
					for i = 0 to n - 1 do
						boutons.(i) <- {titre = titres.(i); orx = i * 37; ory = 0; hauteur = 30;
							largeur = 37; couleur = BLANC; action = (fun () -> ()); bas = false}
					done;
					set_actions boutons;
					cube.boutons1 <- boutons;
;;

let gestion_boutons1 cube is_down mousex mousey =
	for i = 0 to vect_length cube.boutons1 - 1 do
		gestion_bouton cube.boutons1.(i) is_down mousex mousey
	done;
;;

(* Rangée de 12 boutons au dessus de la précédente, justifiée à droite, pour les mouvements des tranches intermédiaires *)
let cree_boutons1i cube =
	let titres =
		[|
			"ai"; "ai'"; "hi"; "hi'"; "di"; "di'";
			"pi"; "pi'"; "bi"; "bi'"; "gi"; "gi'"
		|]
	in
		let set_actions boutons =
			let (OPS (a, d, h), OPS (a', d', h'), OPS (p, g, b), OPS (p', g', b')) =
				cube.rotations_faces1i
			in
				let v = [|a; a'; h; h'; d; d'; p; p'; b; b'; g; g'|]
				and op_names = [|"ai"; "ai'"; "hi"; "hi'"; "di"; "di'"; "pi"; "pi'"; "bi"; "bi'"; "gi"; "gi'"|]
				in
					for i = 0 to vect_length v - 1 do
						set_action boutons.(i) (fun () -> v.(i) (); print_string (op_names.(i)); dessine_cube cube; flush std_out)
					done
		
		in
			let n = vect_length titres in
				let boutons = make_vect n (bouton "" 0 0 0 0 BLANC (fun () -> ())) in
					for i = 0 to n - 1 do
						boutons.(i) <- {titre = titres.(i); orx = (i + 6) * 37; ory = 30; hauteur = 30;
							largeur = 37; couleur = BLANC; action = (fun () -> ()); bas = false}
					done;
					set_actions boutons;
					cube.boutons1 <- vect_of_list (list_of_vect cube.boutons1 @ list_of_vect boutons);
;;

exception Quitter;;

let cree_boutons cube actions =
	let largeur, hauteur = graphics__text_size "centres"
	and largeur4, hauteur4 = graphics__text_size "angles"
	and largeur5, hauteur5 = graphics__text_size "Résolution simple"
	and largeur6, hauteur6 = graphics__text_size "Résolution complète"
	and largeur1, hauteur1 = graphics__text_size "Quitter"
	and largeur2, hauteur2 = graphics__text_size "Mélanger"
	and largeur7, hauteur7 = graphics__text_size "Restituer"
	and largeur8, hauteur8 = graphics__text_size "Composer"
	and _ = cree_boutons1 cube
	and _ = cree_boutons1i cube
	in
		let bouton_centres = bouton "Centres" 20 (graphics__size_y () - hauteur - 20)
			(largeur + 10) (hauteur + 10) JAUNE actions.(0)
		and bouton_angles = bouton "Angles" 20 (graphics__size_y () - 2 * (hauteur4 + 20))
			(largeur4 + 10) (hauteur4 + 10) JAUNE actions.(1)
		and bouton_cube3 = bouton "Résolution simple" 20 (graphics__size_y () - 3 * (hauteur4 + 20))
			(largeur5 + 10) (hauteur5 + 10) JAUNE actions.(2)
		and bouton_cube4 = bouton "Résolution complète" 20 (graphics__size_y () - 4 * (hauteur4 + 20))
			(largeur6 + 10) (hauteur6 + 10) JAUNE actions.(3)
		and bouton_melanger = bouton "Mélanger" ((graphics__size_x () - largeur2) / 2 - 10) (graphics__size_y () - hauteur2 - 20)
			(largeur2 + 10) (hauteur2 + 10) JAUNE actions.(4)
		and bouton_restituer = bouton "Restituer" ((graphics__size_x () - largeur7) / 2 - 10) (graphics__size_y () - 2 * (hauteur7 + 20))
			(largeur7 + 10) (hauteur7 + 10) JAUNE actions.(5)
		and bouton_composer = bouton "Composer" ((graphics__size_x () - largeur8) / 2 - 10) (graphics__size_y () - 3 * (hauteur8 + 20))
			(largeur8 + 10) (hauteur8 + 10) JAUNE actions.(6)
		and bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
			(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
		in
			[bouton_quitter; bouton_centres; bouton_angles; bouton_cube3; bouton_cube4; bouton_melanger; bouton_restituer; bouton_composer]
;;

let dessine_boutons liste_boutons =
	do_list dessine_bouton liste_boutons
;;

let gestion_boutons liste_boutons is_down mousex mousey =
	do_list (fun b -> gestion_bouton b is_down mousex mousey) liste_boutons
;;

(*-------------------------------------------------------------------------------------------*)
(* 18 : SAISIE D'UN MOUVEMENT *)
(*-------------------------------------------------------------------------------------------*)

exception Fin_de_recherche;;(* on a cliqué sur une face de minicube dans le panneau gauche *)

(* affichage en gris d'un minicube *)
let griser plan centre =
	let c, d, f = faces centre in
		for i = 0 to vect_length d - 1 do
			let v = d.(i) in
				draw ((map_vect (prj plan v)
						f.(i)),
					couleur_rvb_de_couleur GRIS)
		done
;;

let est_vide_a_gauche cube v = not est_axe v && fun_of_mv1 cube.mvi.mv1 v <> matrice_nulle;;

let translate (ox, oy, ux, uy) h = (ox + h, oy, ux, uy);;

let affiche_mvi cube plan largeur =
	let plan1 = translate plan (largeur / 2) in
		do_list (griser plan1) indices;
		let p = cube.context1.matrice in
			let f i =
				if est_vide_a_gauche cube i then griser plan (i /:/ p)
				else (affiche1 plan id cube.context1 (i /:/ p) largeur);
				if fun_of_mv1 cube.mvi.mv1 i <> matrice_nulle then
					affiche1 plan1 (transpose p /./ fun_of_mv1 cube.mvi.mv1 i /./ p) cube.context1 (i /:/ p) largeur
			in
				do_list f (select (fun i -> not (est_axe i)) indices)
;;

(* pour les clics dans les faces des minicubes *)
(* est_dans_poly p x renvoie 'true' ssi le point 'x' est intérieur au quadrilatère convexe 'p'*)
let est_dans_poly p x =
	let prefix /-/ (a1, b1) (a2, b2) = (a1 - a2, b1 - b2)
	and det (a1, b1) (a2, b2) = a1 * b2 - a2 * b1
	in
		let p0 = p.(0) /-/ x
		and p1 = p.(1) /-/ x
		and p2 = p.(2) /-/ x
		and p3 = p.(3) /-/ x
		in
			det p0 p1 * det p1 p2 > 0
			&& det p1 p2 * det p2 p3 > 0
			&& det p2 p3 * det p3 p0 > 0
;;

let gestion_gauche_droite cube largeur =
	let choix_a_gauche = ref NIL and choix_a_droite = ref NIL
	and action plan mousex mousey =
		let x = ref NIL in
			try
				do_list (
					fun i ->
									let (c, d, f) = faces i in
										for j = 0 to vect_length d - 1 do
											let face = map_vect (prj plan d.(j)) f.(j) in
												if est_dans_poly face (mousex, mousey) then (x := COUPLE (c, d.(j)); raise Fin_de_recherche)
										done
				)
				(select (fun i -> not (est_axe i)) indices);
				NIL
			with Fin_de_recherche -> !x
	in
		let gestion_a_gauche plan mousex mousey = (
				if mousex < largeur / 2 then (
						choix_a_gauche := action plan mousex mousey;
					)
			)
		and gestion_a_droite plan mousex mousey = (
				if mousex >= largeur / 2 then
					(
						choix_a_droite := action (translate plan (largeur / 2)) mousex mousey;
						let p = cube.context1.matrice in
							match !choix_a_droite with
								| COUPLE (w1, w2) ->
											(
												let est_vide_a_droite v = for_all (fun i -> i /:/ fun_of_mv1 cube.mvi.mv1 i <> v) (select (fun i -> not (est_axe i)) indices)
												in
													if est_vide_a_droite (w1 /:/ transpose p) then
														(
															if not est_axe w1 then
																(match !choix_a_gauche with
																		| COUPLE (v1, v2) ->
																					(try
																							let mat = hd (select (fun m -> v1 /:/ m = w1 && v2 /:/ m = w2) groupe_du_cube)
																							in
																								(let f i = if i /:/ p <> v1 then fun_of_mv1 cube.mvi.mv1 i else p /./ mat /./ transpose p in
																										cube.mvi.mv1 <- map (fun i -> (i, f i)) indices;
																								);
																								choix_a_gauche := NIL
																						with Failure "hd" -> ());
																		| NIL -> ()
																);
														)
													else
														(
															cube.mvi.mv1 <- map (fun i -> (i, let m = fun_of_mv1 cube.mvi.mv1 i in if i /:/ m /:/ p = w1 then matrice_nulle else m)) indices;
															choix_a_gauche := NIL;
														)
											)
								| NIL -> ()
					);
			)
		in
			(gestion_a_gauche, gestion_a_droite)
;;

(*-------------------------------------------------------------------------------------------*)
(* 19 : BOUCLE DE SAISIE D'UN MOUVEMENT DU CUBE *)
(*-------------------------------------------------------------------------------------------*)

let tout_vide_a_gauche cube = for_all (est_vide_a_gauche cube) (select (fun i -> not est_axe i) indices);;

let boucle_saisie cube s =
	graphics__open_graph s;
	let (largeur, hauteur) = (graphics__size_x ()), (graphics__size_y ())
	in
		let plan = (largeur / 4, hauteur / 2, largeur / 80, largeur / 80)
		in let (gestion_a_gauche, gestion_a_droite) = gestion_gauche_droite cube largeur
			in
				graphics__set_window_title "Composer un cube";
				affiche_mvi cube plan largeur;
				graphics__set_color graphics__black;
				graphics__moveto (largeur / 2) 0;
				graphics__lineto (largeur / 2) hauteur; (* cloison entre les deux panneaux *)
				printf__printf "\n-----------------------------------------------------------\n";
				printf__printf "COMPOSITION D'UN MOUVEMENT:\n\n";
				printf__printf "- Pour transférer un minicube de gauche à droite, cliquer sur une de ses faces puis cliquer à droite sur la destination de cette face (une face grise).\n";
				printf__printf "- Pour ramener un minicube de droite à sa place à gauche, cliquer sur une de ses faces.\n";
				printf__printf "- Tant que tous les minicubes n'ont pas été transférés à droite, on peut abandonner en cliquant sur 'Quitter'\n\n";
				printf__printf "- Quand tous les minicubes sont transférés à droite,\n";
				printf__printf "-- la rotation totale des coins est vérifiée: une rotation totale non nulle implique une erreur dans la copie d'un cube correct.\n";
				printf__printf "-- les signatures des permutations des centres et des coins sont vérifiées: des signatures différentes impliquent une erreur dans la copie d'un cube correct.\n";
				printf__printf "-- il est encore possible d'effectuer des corrections.\n\n";
				print_newline ();
				try
					let largeur1, hauteur1 = graphics__text_size "Quitter"
					in
						let bouton_quitter = bouton "Quitter" (graphics__size_x () - largeur1 - 20) (graphics__size_y () - hauteur1 - 20)
							(largeur1 + 10) (hauteur1 + 10) JAUNE (fun () -> raise Quitter)
						in
							dessine_bouton bouton_quitter;
							while true do
								(let status = graphics__wait_next_event [graphics__Button_down] in
										let mousex = status.graphics__mouse_x and mousey = status.graphics__mouse_y
										in gestion_bouton bouton_quitter true mousex mousey
								);
								let status = graphics__wait_next_event [graphics__Button_up] in
									let mousex = status.graphics__mouse_x and mousey = status.graphics__mouse_y
									in
										gestion_bouton bouton_quitter false mousex mousey;
										gestion_a_gauche plan mousex mousey;
										gestion_a_droite plan mousex mousey;
										affiche_mvi cube plan largeur;
										if tout_vide_a_gauche cube then
											(
												printf__printf "TOUS PLACÉS !\n";
												print_newline ();
												let sco = sign (select est_coin indices) (sur cube.mvi.mv1)
												and sce = sign (select est_centre indices) (sur cube.mvi.mv1)
												and rtc1 = rtc cube.mvi.mv1
												in
													printf__printf "signature de la permutation des coins : %d\n" sco;
													printf__printf "signature de la permutation des centres : %d\n" sce;
													printf__printf "rotation totale des coins : %d\n" rtc1;
													print_newline ();
													if sco <> sce || rtc1 <> 0 then (
															printf__printf "MOUVEMENT INCORRECT : \n";
															if sco <> sce then printf__printf "la permutation des centres et celle des coins devraient avoir même signature\n";
															if rtc1 <> 0 then printf__printf "la rotation totale des coins devrait être nulle";
															print_newline ();
														);
											);
							done;
				with Quitter ->
								graphics__close_graph ();
								if tout_vide_a_gauche cube then
									(
										cube.mouvement1.mv1 <- cube.mvi.mv1;
									)
								else (
										printf__printf "composition de mouvement inachevée : ABANDON";
										print_newline ();
									);
;;

(*-------------------------------------------------------------------------------------------*)
(* 20 : BOUCLE DE MANIPULATION DU CUBE 4x4 *)
(*-------------------------------------------------------------------------------------------*)

let boucle1 cube actions =
	try
		dessine_cube cube;
		let liste_boutons = cree_boutons cube actions
		in
			dessine_boutons liste_boutons;
			dessine_boutons1 cube;
			while true do
				let status = graphics__wait_next_event [graphics__Button_down; graphics__Button_up; graphics__Mouse_motion] in
					let mousex = status.graphics__mouse_x
					and mousey = status.graphics__mouse_y
					and is_down = status.graphics__button
					in
						gestion_boutons liste_boutons is_down mousex mousey;
						gestion_boutons1 cube is_down mousex mousey;
			done;
	with Quitter -> graphics__close_graph ()
;;

(*-------------------------------------------------------------------------------------------*)
(* 21 : BOUCLE PRINCIPALE *)
(*-------------------------------------------------------------------------------------------*)

let rec boucle cube =
	(let (ox, oy, _, _) = cube.repere1.plan in
			let (sx, sy) = (string_of_int (2 * ox), string_of_int (2 * oy)) in
				graphics__open_graph (" " ^ sx ^ "x" ^ sy);
				graphics__set_window_title "Résoudre le cube"
	);
	try
		boucle1 cube
		[|
			((* centres *)
				fun () ->
								nbqt := 0;
								lo := "";
								matr := cube.context1.matrice;
								print_string "\n(*-----------------------------------------------------------*)\n";
								print_newline ();
								regrouper_les_centres cube;
								dessine_cube cube;
								printf__printf "\n(* CONSTRUIRE LES CENTRES: %s *)\n" (nbqdt !nbqt);
								print_string !lo;
								print_newline ()
			);
			((* angles *)
				fun () ->
								nbqt := 0;
								lo := "";
								matr := cube.context1.matrice;
								print_string "\n(*-----------------------------------------------------------*)\n";
								print_newline ();
								apparier_les_angles cube;
								dessine_cube cube;
								printf__printf "\n(* APPARIER LES ANGLES: %s *)\n" (nbqdt !nbqt);
								print_string !lo;
								print_newline ()
			);
			((* résolution simple *)
				fun () ->
								nbqt := 0;
								lo := "";
								matr := cube.context1.matrice;
								print_string "\n(*-----------------------------------------------------------*)\n";
								print_newline ();
								enregistrer_mouv cube.mouvement1.mv1 "mouv4444";
								let _ = resoudre_le_cube_4x4 cube
								in
									dessine_cube cube;
									print_newline ();
			);
			((* résolution complète *)
				fun () ->
								nbqt := 0;
								lo := "";
								matr := cube.context1.matrice;
								print_string "\n(*-----------------------------------------------------------*)\n";
								enregistrer_mouv cube.mouvement1.mv1 "mouv4444";
								print_newline ();
								let _ = resoudre_completement_le_cube_4x4 cube
								in
									dessine_cube cube;
									print_newline ();
			);
			((* mélanger *)
				fun () ->
								nbqt := 0;
								lo := "";
								matr := cube.context1.matrice;
								random__init (int_of_float (10000. *. sys__time ()));
								cube.mouvement1.mv1 <- mv1_rubik_r ();
								dessine_cube cube
			);
			((* restituer *)
				fun () ->
								cube.mouvement1.mv1 <- lire_mouv "mouv4444";
								dessine_cube cube
			);
			((* composer *)
				fun () -> let hauteur = 500 and largeur = 1200
								in
									let l = string_of_int largeur and h = string_of_int hauteur in
										graphics__close_graph ();
										boucle_saisie cube (" " ^ l ^ "x" ^ h);
										boucle cube
			);
		|]
	with graphics__Graphic_failure s -> ()
;;

let mouvement = {mv1 = lire_mouv "mouv4444"};;
let context = {matrice = id};;
let repere = {plan = (333, 400, 20, 20)};;
let anime = ref true;;
let cube = nouveau_cube mouvement context repere anime;;
boucle cube;;